<?php


// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('HAROMINKA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('HAROMINKA_PLUGIN_URL', plugin_dir_url(__FILE__));

// Register activation hook
register_activation_hook(__FILE__, 'harominka_plugin_activate');

// Register deactivation hook
register_deactivation_hook(__FILE__, 'harominka_plugin_deactivate');

/**
 * Plugin activation function
 */
function harominka_plugin_activate() {
    // Create plugin options with default values
    $options = array(
        'image_left' => '',
        'image_right' => '',
    );
    
    add_option('harominka_options', $options);
}

/**
 * Plugin deactivation function
 */
function harominka_plugin_deactivate() {
    // Cleanup if needed
}

/**
 * Enqueue scripts and styles
 */
function harominka_enqueue_scripts() {
    // Enqueue CSS
    wp_enqueue_style('harominka-style', HAROMINKA_PLUGIN_URL . 'css/harominka-style.css', array(), '1.0.0');
    
    // Enqueue JavaScript
    wp_enqueue_script('harominka-script', HAROMINKA_PLUGIN_URL . 'js/harominka-script.js', array('jquery'), '1.0.0', true);
}
add_action('wp_enqueue_scripts', 'harominka_enqueue_scripts');

/**
 * Enqueue admin scripts and styles
 */
function harominka_admin_enqueue_scripts($hook) {
    // Only load on plugin admin page
    if ($hook != 'toplevel_page_harominka-settings') {
        return;
    }
    
    // Enqueue media uploader
    wp_enqueue_media();
    
    // Enqueue admin CSS
    wp_enqueue_style('harominka-admin-style', HAROMINKA_PLUGIN_URL . 'css/harominka-admin-style.css', array(), '1.0.0');
    
    // Enqueue admin JavaScript
    wp_enqueue_script('harominka-admin-script', HAROMINKA_PLUGIN_URL . 'js/harominka-admin-script.js', array('jquery'), '1.0.0', true);
}
add_action('admin_enqueue_scripts', 'harominka_admin_enqueue_scripts');

/**
 * Add admin menu - főmenübe helyezve
 */
function harominka_add_admin_menu() {
    add_menu_page(
        'Harominka Image Comparison', 
        'Harominka Images', 
        'manage_options', 
        'harominka-settings', 
        'harominka_settings_page',
        'dashicons-images-alt2',
        30
    );
}
add_action('admin_menu', 'harominka_add_admin_menu');

/**
 * Register settings
 */
function harominka_register_settings() {
    register_setting('harominka_settings_group', 'harominka_options');
}
add_action('admin_init', 'harominka_register_settings');

/**
 * Admin settings page
 */
function harominka_settings_page() {
    // Get saved options
    $options = get_option('harominka_options');
    
    // Default values
    $image_left = isset($options['image_left']) ? $options['image_left'] : '';
    $image_right = isset($options['image_right']) ? $options['image_right'] : '';
    
    // Image previews
    $left_image_url = !empty($image_left) ? wp_get_attachment_url($image_left) : '';
    $right_image_url = !empty($image_right) ? wp_get_attachment_url($image_right) : '';
    
    ?>
    <div class="wrap">
        <h1><?php _e('Harominka Image Comparison Settings', 'harominka-image-comparison'); ?></h1>
        
        <form method="post" action="options.php">
            <?php settings_fields('harominka_settings_group'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="harominka_left_image"><?php _e('Left Image (70%)', 'harominka-image-comparison'); ?></label>
                    </th>
                    <td>
                        <input type="hidden" id="harominka_left_image" name="harominka_options[image_left]" value="<?php echo esc_attr($image_left); ?>" />
                        <div class="harominka-image-preview-wrapper">
                            <?php if (!empty($left_image_url)) : ?>
                                <img src="<?php echo esc_url($left_image_url); ?>" class="harominka-image-preview" alt="Left image preview" />
                            <?php endif; ?>
                        </div>
                        <input type="button" class="button harominka-upload-image" data-target="harominka_left_image" value="<?php _e('Upload Image', 'harominka-image-comparison'); ?>" />
                        <input type="button" class="button harominka-remove-image" data-target="harominka_left_image" value="<?php _e('Remove Image', 'harominka-image-comparison'); ?>" <?php echo empty($left_image_url) ? 'style="display:none"' : ''; ?> />
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="harominka_right_image"><?php _e('Right Image (30%)', 'harominka-image-comparison'); ?></label>
                    </th>
                    <td>
                        <input type="hidden" id="harominka_right_image" name="harominka_options[image_right]" value="<?php echo esc_attr($image_right); ?>" />
                        <div class="harominka-image-preview-wrapper">
                            <?php if (!empty($right_image_url)) : ?>
                                <img src="<?php echo esc_url($right_image_url); ?>" class="harominka-image-preview" alt="Right image preview" />
                            <?php endif; ?>
                        </div>
                        <input type="button" class="button harominka-upload-image" data-target="harominka_right_image" value="<?php _e('Upload Image', 'harominka-image-comparison'); ?>" />
                        <input type="button" class="button harominka-remove-image" data-target="harominka_right_image" value="<?php _e('Remove Image', 'harominka-image-comparison'); ?>" <?php echo empty($right_image_url) ? 'style="display:none"' : ''; ?> />
                    </td>
                </tr>
            </table>
            
            <div class="harominka-shortcode-info">
                <h2><?php _e('Shortcode Usage', 'harominka-image-comparison'); ?></h2>
                <p><?php _e('Use the following shortcode to display the image comparison on your page or post:', 'harominka-image-comparison'); ?></p>
                <code>[harominka_comparison]</code>
            </div>
            
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * Harominka image comparison shortcode
 */
function harominka_comparison_shortcode($atts) {
    // Get options
    $options = get_option('harominka_options');
    
    // Check if images are set
    if (empty($options['image_left']) || empty($options['image_right'])) {
        return '<p>' . __('Please set both images in the Harominka Image Comparison settings.', 'harominka-image-comparison') . '</p>';
    }
    
    // Get image URLs
    $left_image_url = wp_get_attachment_url($options['image_left']);
    $right_image_url = wp_get_attachment_url($options['image_right']);
    
    // Generate unique ID
    $unique_id = 'harominka-' . uniqid();
    
    // Build output
    $output = '<div class="harominka-container" id="' . esc_attr($unique_id) . '">';
    $output .= '<div class="harominka-image-wrapper">';
    $output .= '<div class="harominka-image harominka-image-left" style="background-image: url(' . esc_url($left_image_url) . ')">';
    $output .= '<span class="harominka-icon"><svg fill="none" height="24" viewBox="0 0 24 24" width="24" xmlns="http://www.w3.org/2000/svg" id="fi_18980501"><path clip-rule="evenodd" d="m12.3464 7.50716c.2722-.31222.746-.34467 1.0582-.07248l4.5883 4.00002c.1633.1424.2571.3486.2571.5653 0 .2168-.0938.4229-.2571.5653l-4.5883 4c-.3122.2722-.786.2398-1.0582-.0724-.2722-.3123-.2397-.786.0725-1.0582l3.0795-2.6847h-8.9984c-.41421 0-.75-.3358-.75-.75s.33579-.75.75-.75h8.9984l-3.0795-2.68466c-.3122-.27219-.3447-.74596-.0725-1.05818z" fill="#363853" fill-rule="evenodd"></path></svg></span>';
    $output .= '</div>';
    $output .= '<div class="harominka-image harominka-image-right" style="background-image: url(' . esc_url($right_image_url) . ')">';
    $output .= '<span class="harominka-icon"><svg fill="none" height="24" viewBox="0 0 24 24" width="24" xmlns="http://www.w3.org/2000/svg" id="fi_18980501"><path clip-rule="evenodd" d="m12.3464 7.50716c.2722-.31222.746-.34467 1.0582-.07248l4.5883 4.00002c.1633.1424.2571.3486.2571.5653 0 .2168-.0938.4229-.2571.5653l-4.5883 4c-.3122.2722-.786.2398-1.0582-.0724-.2722-.3123-.2397-.786.0725-1.0582l3.0795-2.6847h-8.9984c-.41421 0-.75-.3358-.75-.75s.33579-.75.75-.75h8.9984l-3.0795-2.68466c-.3122-.27219-.3447-.74596-.0725-1.05818z" fill="#363853" fill-rule="evenodd"></path></svg></span>';
    $output .= '</div>';
    $output .= '</div>';
    $output .= '</div>';
    
    return $output;
}
add_shortcode('harominka_comparison', 'harominka_comparison_shortcode');